import tkinter as tk
from tkinter import messagebox
import config

class Application (tk.Frame):
	def __init__(self, master = None):
		tk.Frame.__init__(self, master)
		self.grid (sticky = tk.N + tk.S + tk.E + tk.W)
		self.createWidgets()
		self.loadcfg ()
		self.loadlevel (None)  # Load the default level.

	def createWidgets (self):
		top = self.winfo_toplevel()
		top.rowconfigure (0, weight = 1)
		top.columnconfigure (0, weight = 1)
		self.rowconfigure (0, weight = 1)
		self.columnconfigure (0, weight = 1)

		# "Explicit" variable declarations
		self.varThisLevel = tk.StringVar()
		self.varThisLevel.set (config.LevelNames [config.defaultlevel])
		self.varNumLevel = tk.IntVar()
		self.varThisRoom = tk.IntVar()
		self.varAssetGroup = tk.StringVar()
		self.varLevelCanvasHeight = tk.IntVar()
		self.varLevelCanvasHeight.set (96)
		self.varNumRooms = tk.IntVar()
		self.varRoomTagmap = {}  # It's going to be a 2-dimensional Dictionary.
		self.varMusic = tk.StringVar()
		self.varMusic.set (config.defaultmusicfix)
		self.varConsolidate = tk.StringVar()
		self.varConsolidate.set (config.defaultconsolidation)
		self.varUnsaved = False

		# Image objects
		#self.imgButtonDelete = tk.PhotoImage (file = config.assetpath + "Other/ButtonDelete.png")
		self.imgRoom00 = tk.PhotoImage()
		self.imgRoom01 = tk.PhotoImage()
		self.imgRoom02 = tk.PhotoImage()
		self.imgRoom03 = tk.PhotoImage()
		self.imgRoom04 = tk.PhotoImage()
		self.imgRoom05 = tk.PhotoImage()
		self.imgRoom06 = tk.PhotoImage()
		self.imgRoom07 = tk.PhotoImage()
		self.imgRoom08 = tk.PhotoImage()
		self.imgRoom09 = tk.PhotoImage()
		self.imgRoom0a = tk.PhotoImage()
		self.imgRoom0b = tk.PhotoImage()
		self.imgRoom0c = tk.PhotoImage()
		self.imgRoom0d = tk.PhotoImage()
		self.imgRoom0e = tk.PhotoImage()
		self.imgRoom0f = tk.PhotoImage()
		self.imgRoom10 = tk.PhotoImage()
		self.imgRoom11 = tk.PhotoImage()
		self.imgRoom12 = tk.PhotoImage()
		self.imgRoom13 = tk.PhotoImage()
		self.imgRoom14 = tk.PhotoImage()
		self.imgRoomEnd = tk.PhotoImage()

		# Scrollbars
		self.yScrollLevel = tk.Scrollbar (self, orient = tk.VERTICAL)
		self.yScrollLevel.grid (row = 0, column = 1, rowspan = 4, sticky = tk.N + tk.S)
		self.yScrollRooms = tk.Scrollbar (self, orient = tk.VERTICAL)
		self.yScrollRooms.grid (row = 0, column = 3, rowspan = 4, sticky = tk.N + tk.S)

		# The level-number selector
		self.optLevel = tk.OptionMenu (self, self.varThisLevel, *config.LevelNames, command = self.dochangelevels)
		self.optLevel.config (font = config.font, bg = config.selectorbgcolor, fg = config.selectorfgcolor, width = 8)
		self.optLevel.grid (row = 2, column = 4, columnspan = 2, pady = 10)

		# The visible frames
		self.frameLevel = tk.LabelFrame (self, font = config.fontframeheader, labelanchor = "n", text = "Level", bg = config.windowbgcolor, fg = config.windowfgcolor)
		self.frameLevel.grid (row = 0, column = 0, rowspan = 4, sticky = tk.N)
		self.frameRooms = tk.LabelFrame (self, font = config.fontframeheader, labelanchor = "n", text = "Rooms", bg = config.windowbgcolor, fg = config.windowfgcolor)
		self.frameRooms.grid (row = 0, column = 2, rowspan = 4)
		self.frameCommands = tk.LabelFrame (self, font = config.fontframeheader, labelanchor = "n", text = "Commands", bg = config.windowbgcolor, fg = config.windowfgcolor)
		self.frameCommands.grid (row = 0, column = 4, columnspan = 2)
		self.frameOptions = tk.LabelFrame (self, font = config.fontframeheader, labelanchor = "n", text = "Options", bg = config.windowbgcolor, fg = config.windowfgcolor)
		self.frameOptions.grid (row = 1, column = 4, columnspan = 2)

		# Canvases to display the current Level and Rooms list
		self.canvasLevel = tk.Canvas (self.frameLevel, bg = config.canvasbgcolor, height = min (int (config.canvasheight), self.varLevelCanvasHeight.get()), width = 560, scrollregion = [0, 0, 559, self.varLevelCanvasHeight.get() - 1], yscrollcommand = self.yScrollLevel.set)
		self.canvasLevel.grid()
		self.yScrollLevel ["command"] = self.canvasLevel.yview  # Needed to make the scrollbar moveable
		self.canvasLevel.bind ("<Button-1>", self.dopopup)  # Can left-click on the End Room
		self.canvasLevel.bind ("<Button-3>", self.dopopup)  # Can right-click on the End Room
		self.canvasRooms = tk.Canvas (self.frameRooms, height = config.canvasheight, width = 324, scrollregion = [0, 0, 323, 4043], yscrollcommand = self.yScrollRooms.set)
		self.canvasRooms.grid()
		self.yScrollRooms ["command"] = self.canvasRooms.yview

		# Command buttons
		self.btnHelp = tk.Button (self.frameCommands, font = config.font, text = "Instructions", width = 10, bg = config.buttonbgcolor, fg = config.buttonfgcolor, disabledforeground = config.buttongreyoutcolor, state = tk.NORMAL, command = self.dohelp)
		self.btnHelp.grid (row = 0, sticky = tk.N + tk.S + tk.E + tk.W)
		self.btnSave = tk.Button (self.frameCommands, font = config.font, text = "Save", width = 10, bg = config.buttonbgcolor, fg = config.buttonfgcolor, disabledforeground = config.buttongreyoutcolor, state = tk.DISABLED, command = self.dosave)  # "Greyed-out" when there are no changes to save.  We'll use this to know whether to flag a warning about unsaved changes and so on.
		self.btnSave.grid (row = 1, sticky = tk.N + tk.S + tk.E + tk.W)
		self.btnBackup = tk.Button (self.frameCommands, font = config.font, text = "Backup", width = 10, bg = config.buttonbgcolor, fg = config.buttonfgcolor, disabledforeground = config.buttongreyoutcolor, state = tk.NORMAL, command = self.dobackup)
		self.btnBackup.grid (row = 2, sticky = tk.N + tk.S + tk.E + tk.W)
		self.btnMake = tk.Button (self.frameCommands, font = config.font, text = "Make .rom", width = 10, bg = config.buttonbgcolor, fg = config.buttonfgcolor, disabledforeground = config.buttongreyoutcolor, state = tk.NORMAL, command = self.domake)
		self.btnMake.grid (row = 3, sticky = tk.N + tk.S + tk.E + tk.W)
		self.btnPlay = tk.Button (self.frameCommands, font = config.font, text = "Play .rom", width = 10, bg = config.buttonbgcolor, fg = config.buttonfgcolor, disabledforeground = config.buttongreyoutcolor, state = tk.NORMAL, command = self.doplay)
		self.btnPlay.grid (row = 4, sticky = tk.N + tk.S + tk.E + tk.W)
		self.btnAbout = tk.Button (self.frameCommands, font = config.font, text = "About", width = 10, bg = config.buttonbgcolor, fg = config.buttonfgcolor, disabledforeground = config.buttongreyoutcolor, state = tk.NORMAL, command = self.doabout)
		self.btnAbout.grid (row = 5, sticky = tk.N + tk.S + tk.E + tk.W)
		self.btnQuit = tk.Button (self.frameCommands, font = config.font, text = "Quit", width = 10, bg = config.buttonbgcolor, fg = config.buttonfgcolor, disabledforeground = config.buttongreyoutcolor, state = tk.NORMAL, command = self.doquit)
		self.btnQuit.grid (row = 6, sticky = tk.N + tk.S + tk.E + tk.W)

		# Checkboxes
		self.chkMusic = tk.Checkbutton (self.frameOptions, font = config.font, anchor = tk.W, text = "Music Fix", bg = config.canvasbgcolor, variable = self.varMusic, offvalue = ";\n", onvalue = "< " + config.workingpath + "musicfix.txt\n")  # The canvas color is used because we're keeping the text black here.  Otherwise the check in the checkbox can't be seen.
		self.chkMusic.grid (row = 0, sticky = tk.E + tk.W)
		self.chkConsolidate = tk.Checkbutton (self.frameOptions, font=config.font, anchor = tk.W, text = "Move Pointer Tables", bg = config.canvasbgcolor, variable = self.varConsolidate, offvalue = ";\n", onvalue = "< " + config.workingpath + "consolidate.txt\n")
		self.chkConsolidate.grid (row = 1, sticky = tk.E + tk.W)

                # "Number of rooms in use" labels
		self.lblNumRooms = tk.Label (self, font = config.font, anchor = tk.S + tk.E, textvariable = self.varNumRooms, bg = config.windowbgcolor, fg = config.windowfgcolor)
		self.lblNumRooms.grid (row = 3, column = 4, sticky = tk.E + tk.S)
		self.lblMaxNumRooms = tk.Label (self, font = config.font, anchor = tk.S + tk.W, text = "Rooms:\n / 1798", bg = config.windowbgcolor, fg = config.windowfgcolor)
		self.lblMaxNumRooms.grid (row = 3, column = 5, sticky = tk.W + tk.S)

		# These images will always be populated on the Rooms canvas.  The only thing that will change is what image they display.
		# They are contained in buttons, which in turn are contained in canvas window objects.
		self.ID00 = tk.Button (self.frameRooms, image = self.imgRoom00, height = 16, width = 318, bg = "black", command = self.doaddroom00)
		self.canvasRooms.create_window (0, 0, anchor = tk.N + tk.W, window = self.ID00)
		self.ID01 = tk.Button (self.frameRooms, image = self.imgRoom01, height = 16, width = 318, bg = "black", command = self.doaddroom01)
		self.canvasRooms.create_window (0, 22, anchor = tk.N + tk.W, window = self.ID01)
		self.ID02 = tk.Button (self.frameRooms, image = self.imgRoom02, height = 16, width = 318, bg = "black", command = self.doaddroom02)
		self.canvasRooms.create_window (0, 44, anchor = tk.N + tk.W, window = self.ID02)
		self.ID03 = tk.Button (self.frameRooms, image = self.imgRoom03, height = 16, width = 318, bg = "black", command = self.doaddroom03)
		self.canvasRooms.create_window (0, 66, anchor = tk.N + tk.W, window = self.ID03)
		self.ID04 = tk.Button (self.frameRooms, image = self.imgRoom04, height = 16, width = 318, bg = "black", command = self.doaddroom04)
		self.canvasRooms.create_window (0, 88, anchor = tk.N + tk.W, window = self.ID04)
		self.ID05 = tk.Button (self.frameRooms, image = self.imgRoom05, height = 16, width = 318, bg = "black", command = self.doaddroom05)
		self.canvasRooms.create_window (0, 110, anchor = tk.N + tk.W, window = self.ID05)
		self.ID06 = tk.Button (self.frameRooms, image = self.imgRoom06, height = 112, width = 318, bg = "black", command = self.doaddroom06)
		self.canvasRooms.create_window (0, 132, anchor = tk.N + tk.W, window = self.ID06)
		self.ID07 = tk.Button (self.frameRooms, image = self.imgRoom07, height = 336, width = 318, bg = "black", command = self.doaddroom07)
		self.canvasRooms.create_window (0, 250, anchor = tk.N + tk.W, window = self.ID07)
		self.ID08 = tk.Button (self.frameRooms, image = self.imgRoom08, height = 112, width = 318, bg = "black", command = self.doaddroom08)
		self.canvasRooms.create_window (0, 592, anchor = tk.N + tk.W, window = self.ID08)
		self.ID09 = tk.Button (self.frameRooms, image = self.imgRoom09, height = 256, width = 318, bg = "black", command = self.doaddroom09)
		self.canvasRooms.create_window (0, 710, anchor = tk.N + tk.W, window = self.ID09)
		self.ID0a = tk.Button (self.frameRooms, image = self.imgRoom0a, height = 208, width = 318, bg = "black", command = self.doaddroom0a)
		self.canvasRooms.create_window (0, 972, anchor = tk.N + tk.W, window = self.ID0a)
		self.ID0b = tk.Button (self.frameRooms, image = self.imgRoom0b, height = 160, width = 318, bg = "black", command = self.doaddroom0b)
		self.canvasRooms.create_window (0, 1186, anchor = tk.N + tk.W, window = self.ID0b)
		self.ID0c = tk.Button (self.frameRooms, image = self.imgRoom0c, height = 128, width = 318, bg = "black", command = self.doaddroom0c)
		self.canvasRooms.create_window (0, 1352, anchor = tk.N + tk.W, window = self.ID0c)
		self.ID0d = tk.Button (self.frameRooms, image = self.imgRoom0d, height = 256, width = 318, bg = "black", command = self.doaddroom0d)
		self.canvasRooms.create_window (0, 1486, anchor = tk.N + tk.W, window = self.ID0d)
		self.ID0e = tk.Button (self.frameRooms, image = self.imgRoom0e, height = 208, width = 318, bg = "black", command = self.doaddroom0e)
		self.canvasRooms.create_window (0, 1748, anchor = tk.N + tk.W, window = self.ID0e)
		self.ID0f = tk.Button (self.frameRooms, image = self.imgRoom0f, height = 288, width = 318, bg = "black", command = self.doaddroom0f)
		self.canvasRooms.create_window (0, 1962, anchor = tk.N + tk.W, window = self.ID0f)
		self.ID10 = tk.Button (self.frameRooms, image = self.imgRoom10, height = 304, width = 318, bg = "black", command = self.doaddroom10)
		self.canvasRooms.create_window (0, 2256, anchor = tk.N + tk.W, window = self.ID10)
		self.ID11 = tk.Button (self.frameRooms, image = self.imgRoom11, height = 304, width = 318, bg = "black", command = self.doaddroom11)
		self.canvasRooms.create_window (0, 2566, anchor = tk.N + tk.W, window = self.ID11)
		self.ID12 = tk.Button (self.frameRooms, image = self.imgRoom12, height = 400, width = 318, bg = "black", command = self.doaddroom12)
		self.canvasRooms.create_window (0, 2876, anchor = tk.N + tk.W, window = self.ID12)
		self.ID13 = tk.Button (self.frameRooms, image = self.imgRoom13, height = 432, width = 318, bg = "black", command = self.doaddroom13)
		self.canvasRooms.create_window (0, 3282, anchor = tk.N + tk.W, window = self.ID13)
		self.ID14 = tk.Button (self.frameRooms, image = self.imgRoom14, height = 320, width = 318, bg = "black", command = self.doaddroom14)
		self.canvasRooms.create_window (0, 3720, anchor = tk.N + tk.W, window = self.ID14)

		# Popup menu
		self.menuPopup = tk.Menu(self, tearoff = 0)
		self.menuPopup.add_command (label = "Move Insertion Point below this room", command = self.domoveinsertionpoint)
		self.menuPopup.add_command (label = "Delete this room", command = self.dodeleteroom)
		#self.menuPopup.add_separator()
		self.menuPopup.add_command (label = "Cancel")		

	# Code to load a new level from the appropriate sublist in self.varRoomList.
	def loadlevel (self, event):
		# Re-initialize the Level canvas.
		self.canvasLevel.delete ("all")  # Special keyword to delete all the elements from the canvas
		self.varLevelCanvasHeight.set (96)
		self.canvasLevel.config (height = min (int (config.canvasheight), self.varLevelCanvasHeight.get()), scrollregion = [0, 0, 559, self.varLevelCanvasHeight.get() - 1])
		self.varRoomTagmap.clear()

		# Fetch the assets for this level.
		for index, string in enumerate (config.LevelNames):
			if (self.varThisLevel.get() == string):
				self.varNumLevel.set (index)
				self.imgRoom00.config (file = config.AssetList [index] [0])
				self.imgRoom01.config (file = config.AssetList [index] [1])
				self.imgRoom02.config (file = config.AssetList [index] [2])
				self.imgRoom03.config (file = config.AssetList [index] [3])
				self.imgRoom04.config (file = config.AssetList [index] [4])
				self.imgRoom05.config (file = config.AssetList [index] [5])
				self.imgRoom06.config (file = config.AssetList [index] [6])
				self.imgRoom07.config (file = config.AssetList [index] [7])
				self.imgRoom08.config (file = config.AssetList [index] [8])
				self.imgRoom09.config (file = config.AssetList [index] [9])
				self.imgRoom0a.config (file = config.AssetList [index] [10])
				self.imgRoom0b.config (file = config.AssetList [index] [11])
				self.imgRoom0c.config (file = config.AssetList [index] [12])
				self.imgRoom0d.config (file = config.AssetList [index] [13])
				self.imgRoom0e.config (file = config.AssetList [index] [14])
				self.imgRoom0f.config (file = config.AssetList [index] [15])
				self.imgRoom10.config (file = config.AssetList [index] [16])
				self.imgRoom11.config (file = config.AssetList [index] [17])
				self.imgRoom12.config (file = config.AssetList [index] [18])
				self.imgRoom13.config (file = config.AssetList [index] [19])
				self.imgRoom14.config (file = config.AssetList [index] [20])
				self.imgRoomEnd.config (file = config.AssetList [index] [21])

		# Load the End room and insertion point.
		self.IDEnd = self.canvasLevel.create_image (3, 0, anchor = tk.N + tk.W, image = self.imgRoomEnd)
		self.InsertionPoint = self.canvasLevel.create_text (159, 80, anchor = tk.N, font = config.font, text = "Insert below", width = 318)
		self.varRoomTagmap [self.IDEnd] = {"height": 80, "pos": 0}
		self.varRoomTagmap [self.InsertionPoint] = {"height": 16, "pos": 80}

		# Load all the other rooms for this level.
		for room, params in zip (* [iter (self.varRoomList [self.varNumLevel.get()])] * 2):  # Read each pair of elements in the appropriate list.
			if (params != "80"):  # The End room was already loaded onto the canvas.
				self.addroomtolevel (room, params, False)

	# Code to add a room to the Level canvas.  Each room will be represented as a Frame widget drawn on the canvas.
	def addroomtolevel (self, room, params, incrementnumrooms):
		if (self.varNumRooms.get() >= 1798) and (incrementnumrooms):
			tk.messagebox.showerror ("Error", "Total number of rooms is at the limit.")
		else:
			if (incrementnumrooms):
				self.varNumRooms.set (self.varNumRooms.get() + 1)
				self.btnSave.config (state = tk.NORMAL)
				self.varUnsaved = True
			RoomFrame = tk.Frame (self.canvasLevel, bg = config.canvasbgcolor, bd = 0, relief = tk.FLAT)  # We don't want a visible border around this frame.
			RoomFrame.bind ("<Button-3>", self.dopopup)

			# Handle everything unique to the ID of the room we're adding.
			if (room == "0"):
				RoomImage = tk.Button (RoomFrame, image = self.imgRoom00, bg = "black")
				NewRoomHeight = 16 * 1 + 6
				RoomDurationLabel = tk.Label (RoomFrame, font = ("Mono", "10"), text = " Length:", bg = config.canvasbgcolor)
				RoomDurationSlider = tk.Spinbox (RoomFrame, bg = config.canvasbgcolor, bd = 0, font = ("Mono", "10"), from_= 1, to = 63, increment = 1, relief = tk.SOLID, width = 2, command = self.doflagunsaved, exportselection = True, wrap = True)
			elif (room == "1"):
				RoomImage = tk.Button (RoomFrame, image = self.imgRoom01, bg = "black")
				NewRoomHeight = 16 * 1 + 6
				RoomDurationLabel = tk.Label (RoomFrame, font = ("Mono", "10"), text = " Length:", bg = config.canvasbgcolor)
				RoomDurationSlider = tk.Spinbox (RoomFrame, bg = config.canvasbgcolor, bd = 0, font = ("Mono", "10"), from_= 1, to = 63, increment = 1, relief = tk.SOLID, width = 2, command = self.doflagunsaved, exportselection = True, wrap = True)
			elif (room == "2"):
				RoomImage = tk.Button (RoomFrame, image = self.imgRoom02, bg = "black")
				NewRoomHeight = 16 * 1 + 6
				RoomDurationLabel = tk.Label (RoomFrame, font = ("Mono", "10"), text = " Repeat:", bg = config.canvasbgcolor)
				RoomDurationSlider = tk.Spinbox (RoomFrame, bg = config.canvasbgcolor, bd = 0, font = ("Mono", "10"), from_= 0, to = 63, increment = 1, relief = tk.SOLID, width = 2, command = self.doflagunsaved, exportselection = True, wrap = True)
			elif (room == "3"):
				RoomImage = tk.Button (RoomFrame, image = self.imgRoom03, bg = "black")
				NewRoomHeight = 16 * 1 + 6
				RoomDurationLabel = tk.Label (RoomFrame, font = ("Mono", "10"), text = " Length:", bg = config.canvasbgcolor)
				RoomDurationSlider = tk.Spinbox (RoomFrame, bg = config.canvasbgcolor, bd = 0, font = ("Mono", "10"), from_= 1, to = 63, increment = 1, relief = tk.SOLID, width = 2, command = self.doflagunsaved, exportselection = True, wrap = True)
			elif (room == "4"):
				RoomImage = tk.Button (RoomFrame, image = self.imgRoom04, bg = "black")
				NewRoomHeight = 16 * 1 + 6
				RoomDurationLabel = tk.Label (RoomFrame, font = ("Mono", "10"), text = " Length:", bg = config.canvasbgcolor)
				RoomDurationSlider = tk.Spinbox (RoomFrame, bg = config.canvasbgcolor, bd = 0, font = ("Mono", "10"), from_= 1, to = 63, increment = 1, relief = tk.SOLID, width = 2, command = self.doflagunsaved, exportselection = True, wrap = True)
			elif (room == "5"):
				RoomImage = tk.Button (RoomFrame, image = self.imgRoom05, bg = "black")
				NewRoomHeight = 16 * 1 + 6
				RoomDurationLabel = tk.Label (RoomFrame, font = ("Mono", "10"), text = " Length:", bg = config.canvasbgcolor)
				RoomDurationSlider = tk.Spinbox (RoomFrame, bg = config.canvasbgcolor, bd = 0, font = ("Mono", "10"), from_= 1, to = 63, increment = 1, relief = tk.SOLID, width = 2, command = self.doflagunsaved, exportselection = True, wrap = True)
			elif (room == "6"):
				RoomImage = tk.Button (RoomFrame, image = self.imgRoom06, bg = "black")
				NewRoomHeight = 16 * 7 + 6
				RoomDurationLabel = tk.Label (RoomFrame, font = ("Mono", "10"), text = " Repeat:", bg = config.canvasbgcolor)
				RoomDurationSlider = tk.Spinbox (RoomFrame, bg = config.canvasbgcolor, bd = 0, font = ("Mono", "10"), from_= 0, to = 63, increment = 1, relief = tk.SOLID, width = 2, command = self.doflagunsaved, exportselection = True, wrap = True)
			elif (room == "7"):
				RoomImage = tk.Button (RoomFrame, image = self.imgRoom07, bg = "black")
				NewRoomHeight = 16 * 21 + 6
				RoomDurationLabel = tk.Label (RoomFrame, font = ("Mono", "10"), text = " Repeat:", bg = config.canvasbgcolor)
				RoomDurationSlider = tk.Spinbox (RoomFrame, bg = config.canvasbgcolor, bd = 0, font = ("Mono", "10"), from_= 0, to = 63, increment = 1, relief = tk.SOLID, width = 2, command = self.doflagunsaved, exportselection = True, wrap = True)
			elif (room == "8"):
				RoomImage = tk.Button (RoomFrame, image = self.imgRoom08, bg = "black")
				NewRoomHeight = 16 * 7 + 6
				RoomDurationLabel = tk.Label (RoomFrame, font = ("Mono", "10"), text = " Repeat:", bg = config.canvasbgcolor)
				RoomDurationSlider = tk.Spinbox (RoomFrame, bg = config.canvasbgcolor, bd = 0, font = ("Mono", "10"), from_= 0, to = 63, increment = 1, relief = tk.SOLID, width = 2, command = self.doflagunsaved, exportselection = True, wrap = True)
			elif (room == "9"):
				RoomImage = tk.Button (RoomFrame, image = self.imgRoom09, bg = "black")
				NewRoomHeight = 16 * 16 + 6
				RoomDurationLabel = tk.Label (RoomFrame, font = ("Mono", "10"), text = " Repeat:", bg = config.canvasbgcolor)
				RoomDurationSlider = tk.Spinbox (RoomFrame, bg = config.canvasbgcolor, bd = 0, font = ("Mono", "10"), from_= 0, to = 63, increment = 1, relief = tk.SOLID, width = 2, command = self.doflagunsaved, exportselection = True, wrap = True)
			elif (room == "a"):
				RoomImage = tk.Button (RoomFrame, image = self.imgRoom0a, bg = "black")
				NewRoomHeight = 16 * 13 + 6
				RoomDurationLabel = tk.Label (RoomFrame, font = ("Mono", "10"), text = " Repeat:", bg = config.canvasbgcolor)
				RoomDurationSlider = tk.Spinbox (RoomFrame, bg = config.canvasbgcolor, bd = 0, font = ("Mono", "10"), from_= 0, to = 63, increment = 1, relief = tk.SOLID, width = 2, command = self.doflagunsaved, exportselection = True, wrap = True)
			elif (room == "b"):
				RoomImage = tk.Button (RoomFrame, image = self.imgRoom0b, bg = "black")
				NewRoomHeight = 16 * 10 + 6
				RoomDurationLabel = tk.Label (RoomFrame, font = ("Mono", "10"), text = " Repeat:", bg = config.canvasbgcolor)
				RoomDurationSlider = tk.Spinbox (RoomFrame, bg = config.canvasbgcolor, bd = 0, font = ("Mono", "10"), from_= 0, to = 63, increment = 1, relief = tk.SOLID, width = 2, command = self.doflagunsaved, exportselection = True, wrap = True)
			elif (room == "c"):
				RoomImage = tk.Button (RoomFrame, image = self.imgRoom0c, bg = "black")
				NewRoomHeight = 16 * 8 + 6
				RoomDurationLabel = tk.Label (RoomFrame, font = ("Mono", "10"), text = " Repeat:", bg = config.canvasbgcolor)
				RoomDurationSlider = tk.Spinbox (RoomFrame, bg = config.canvasbgcolor, bd = 0, font = ("Mono", "10"), from_= 0, to = 63, increment = 1, relief = tk.SOLID, width = 2, command = self.doflagunsaved, exportselection = True, wrap = True)
			elif (room == "d"):
				RoomImage = tk.Button (RoomFrame, image = self.imgRoom0d, bg = "black")
				NewRoomHeight = 16 * 16 + 6
				RoomDurationLabel = tk.Label (RoomFrame, font = ("Mono", "10"), text = " Repeat:", bg = config.canvasbgcolor)
				RoomDurationSlider = tk.Spinbox (RoomFrame, bg = config.canvasbgcolor, bd = 0, font = ("Mono", "10"), from_= 0, to = 63, increment = 1, relief = tk.SOLID, width = 2, command = self.doflagunsaved, exportselection = True, wrap = True)
			elif (room == "e"):
				RoomImage = tk.Button (RoomFrame, image = self.imgRoom0e, bg = "black")
				NewRoomHeight = 16 * 13 + 6
				RoomDurationLabel = tk.Label (RoomFrame, font = ("Mono", "10"), text = " Repeat:", bg = config.canvasbgcolor)
				RoomDurationSlider = tk.Spinbox (RoomFrame, bg = config.canvasbgcolor, bd = 0, font = ("Mono", "10"), from_= 0, to = 63, increment = 1, relief = tk.SOLID, width = 2, command = self.doflagunsaved, exportselection = True, wrap = True)
			elif (room == "f"):
				RoomImage = tk.Button (RoomFrame, image = self.imgRoom0f, bg = "black")
				NewRoomHeight = 16 * 18 + 6
				RoomDurationLabel = tk.Label (RoomFrame, font = ("Mono", "10"), text = " Repeat:", bg = config.canvasbgcolor)
				RoomDurationSlider = tk.Spinbox (RoomFrame, bg = config.canvasbgcolor, bd = 0, font = ("Mono", "10"), from_= 0, to = 63, increment = 1, relief = tk.SOLID, width = 2, command = self.doflagunsaved, exportselection = True, wrap = True)
			elif (room == "10"):
				RoomImage = tk.Button (RoomFrame, image = self.imgRoom10, bg = "black")
				NewRoomHeight = 16 * 19 + 6
				RoomDurationLabel = tk.Label (RoomFrame, font = ("Mono", "10"), text = " Repeat:", bg = config.canvasbgcolor)
				RoomDurationSlider = tk.Spinbox (RoomFrame, bg = config.canvasbgcolor, bd = 0, font = ("Mono", "10"), from_= 0, to = 63, increment = 1, relief = tk.SOLID, width = 2, command = self.doflagunsaved, exportselection = True, wrap = True)
			elif (room == "11"):
				RoomImage = tk.Button (RoomFrame, image = self.imgRoom11, bg = "black")
				NewRoomHeight = 16 * 19 + 6
				RoomDurationLabel = tk.Label (RoomFrame, font = ("Mono", "10"), text = " Repeat:", bg = config.canvasbgcolor)
				RoomDurationSlider = tk.Spinbox (RoomFrame, bg = config.canvasbgcolor, bd = 0, font = ("Mono", "10"), from_= 0, to = 63, increment = 1, relief = tk.SOLID, width = 2, command = self.doflagunsaved, exportselection = True, wrap = True)
			elif (room == "12"):
				RoomImage = tk.Button (RoomFrame, image = self.imgRoom12, bg = "black")
				NewRoomHeight = 16 * 25 + 6
				RoomDurationLabel = tk.Label (RoomFrame, font = ("Mono", "10"), text = " Repeat:", bg = config.canvasbgcolor)
				RoomDurationSlider = tk.Spinbox (RoomFrame, bg = config.canvasbgcolor, bd = 0, font = ("Mono", "10"), from_= 0, to = 63, increment = 1, relief = tk.SOLID, width = 2, command = self.doflagunsaved, exportselection = True, wrap = True)
			elif (room == "13"):
				RoomImage = tk.Button (RoomFrame, image = self.imgRoom13, bg = "black")
				NewRoomHeight = 16 * 27 + 6
				RoomDurationLabel = tk.Label (RoomFrame, font = ("Mono", "10"), text = " Repeat:", bg = config.canvasbgcolor)
				RoomDurationSlider = tk.Spinbox (RoomFrame, bg = config.canvasbgcolor, bd = 0, font = ("Mono", "10"), from_= 0, to = 63, increment = 1, relief = tk.SOLID, width = 2, command = self.doflagunsaved, exportselection = True, wrap = True)
			elif (room == "14"):
				RoomImage = tk.Button (RoomFrame, image = self.imgRoom14, bg = "black")
				NewRoomHeight = 16 * 20 + 6
				RoomDurationLabel = tk.Label (RoomFrame, font = ("Mono", "10"), text = " Repeat:", bg = config.canvasbgcolor)
				RoomDurationSlider = tk.Spinbox (RoomFrame, bg = config.canvasbgcolor, bd = 0, font = ("Mono", "10"), from_= 0, to = 63, increment = 1, relief = tk.SOLID, width = 2, command = self.doflagunsaved, exportselection = True, wrap = True)
			# To set the value of the Spinbox widget, we have to delete its existing value (it's a string) and insert what we want in its place.
			RoomDurationSlider.delete (0, "end")
			RoomDurationSlider.insert (0, str (int (params, base = 16) % 64))

			# Move downward all the rooms currently below the Insertion Point.  Do this first before displaying the new room.
			for index, roomid in enumerate (self.canvasLevel.find_all()):
				if (int (self.varRoomTagmap [roomid] ["pos"]) > int (self.varRoomTagmap [self.InsertionPoint] ["pos"])):
					self.canvasLevel.move (roomid, 0, NewRoomHeight)
					self.varRoomTagmap [roomid].update ({"pos": NewRoomHeight + self.varRoomTagmap [roomid] ["pos"]})

			# A few more things before we create the other child widgets
			IDRoomFrame = self.canvasLevel.create_window (0, int (self.varRoomTagmap [self.InsertionPoint] ["pos"]) + 16, window = RoomFrame, anchor = tk.N + tk.W)
			self.varLevelCanvasHeight.set (self.varLevelCanvasHeight.get() + NewRoomHeight)  # Stretch the canvas to make room for the new frame.
			RoomImage.grid (row = 0, column = 0)  # Display the frame.

			# Add the "water hazard warning" and "restart point" flags to the new room.
			RoomFlagsLabel = tk.Label (RoomFrame, font = ("Sans", "10"), text = " N/R/!:", bg = config.canvasbgcolor)
			RoomFlagsLabel.grid (row = 0, column = 2, sticky = tk.W)
			RoomFlags = tk.Scale (RoomFrame, bg = config.canvasbgcolor, length = 64, orient = tk.HORIZONTAL, showvalue = 0, from_= 0, to = 512, resolution = 256, width = 16)
			RoomFlags.set (int (params, base = 16) // 256 * 256)
			RoomFlags.grid (row = 0, column = 3, sticky = tk.W)

			# We're going to use a Spinbox widget for the "length/repeat" value rather than a Scale widget, so we need a separate Label widget.
			RoomDurationLabel.grid (row=0, column=6, sticky=tk.W)  # The Label was instantiated in the pseudo-Case block above, because the displayed text depends on the room.
			RoomDurationSlider.grid (row=0, column=7, sticky=tk.W)  # Ditto for the Spinbox, because its minimum value also changes based on the room.
			self.canvasLevel.config (height = min (int (config.canvasheight), self.varLevelCanvasHeight.get()), scrollregion = [0, 0, 559, self.varLevelCanvasHeight.get() - 1])  # Stretch the canvas to display the new room.

			# Final steps
			self.varRoomTagmap [IDRoomFrame] = {"height": NewRoomHeight, "pos": 16 + self.varRoomTagmap [self.InsertionPoint] ["pos"], "room": room, "ptrroom": RoomImage, "ptrflags": RoomFlags, "ptrduration": RoomDurationSlider}
			RoomImage.bind ("<Button-1>", self.dopopup)  # Can click on the room image.
			RoomImage.bind ("<Button-3>", self.dopopup)  # Can right-click on the room image.

	# Code to move the Insertion Point to below a certain room
	def moveinsertionpoint (self, room):
		if (room != self.InsertionPoint):  # There is obviously nothing to do if the insertion point is already there.
			OldInsertionPointPos = self.varRoomTagmap [self.InsertionPoint] ["pos"]
			NewInsertionPointPos = self.varRoomTagmap [room] ["height"] + self.varRoomTagmap [room] ["pos"] - self.varRoomTagmap [self.InsertionPoint] ["height"]
			self.canvasLevel.move (self.InsertionPoint, 0, NewInsertionPointPos - OldInsertionPointPos)
			self.varRoomTagmap [self.InsertionPoint].update ({"pos": NewInsertionPointPos})
			if (NewInsertionPointPos > OldInsertionPointPos):
				# Move upward all the rooms between the insertion point's old position and its new position.
				for index, roomid in enumerate (self.canvasLevel.find_all()):
					if (self.varRoomTagmap [roomid] ["pos"] > OldInsertionPointPos) and (self.varRoomTagmap [roomid] ["pos"] < NewInsertionPointPos):
						self.canvasLevel.move (roomid, 0, 0 - self.varRoomTagmap [self.InsertionPoint] ["height"])  # The arguments for .move are offsets, not absolute coordinates.
						self.varRoomTagmap [roomid].update ({"pos": self.varRoomTagmap [roomid] ["pos"] - self.varRoomTagmap [self.InsertionPoint] ["height"]})
			elif (NewInsertionPointPos < OldInsertionPointPos):
				# Move downward all the rooms between the insertion point's old position and new position.
				for index, roomid in enumerate (self.canvasLevel.find_all()):
					if (self.varRoomTagmap [roomid] ["pos"] > NewInsertionPointPos) and (self.varRoomTagmap [roomid] ["pos"] < OldInsertionPointPos):
						self.canvasLevel.move (roomid, 0, self.varRoomTagmap [self.InsertionPoint] ["height"])  # The arguments for .move are offsets, not absolute coordinates.
						self.varRoomTagmap [roomid].update ({"pos": self.varRoomTagmap [roomid] ["pos"] + self.varRoomTagmap [self.InsertionPoint] ["height"]})
				self.canvasLevel.move (self.InsertionPoint, 0, self.varRoomTagmap [self.InsertionPoint] ["height"])
				self.varRoomTagmap [self.InsertionPoint].update ({"pos": NewInsertionPointPos + self.varRoomTagmap [self.InsertionPoint] ["height"]})

	# Code to delete a room from the level
	def deleteroom (self, room):
		if (room != self.IDEnd) and (room != self.InsertionPoint):  # The End Room and Insertion Point must not be deleted here.
			self.canvasLevel.delete (room)
			# Move upward all the rooms below the room just deleted.
			for index, roomid in enumerate (self.canvasLevel.find_all()):
				if (self.varRoomTagmap [roomid] ["pos"] > self.varRoomTagmap [room] ["pos"]):
					self.canvasLevel.move (roomid, 0, 0 - self.varRoomTagmap [room] ["height"])  # The arguments for .move are offsets, not absolute coordinates.
					self.varRoomTagmap [roomid].update ({"pos": self.varRoomTagmap [roomid] ["pos"] - self.varRoomTagmap [room] ["height"]})
			self.varLevelCanvasHeight.set (self.varLevelCanvasHeight.get() - self.varRoomTagmap [room] ["height"])
			self.canvasLevel.config (height = min (int (config.canvasheight), self.varLevelCanvasHeight.get()), scrollregion = [0, 0, 559, self.varLevelCanvasHeight.get() - 1])  # Reduce the canvas size by the amount that is height of the room just deleted.
			self.varRoomTagmap.pop (room)  # Remove the room from the dictionary.
			self.varNumRooms.set (self.varNumRooms.get() - 1)
			self.btnSave.config(state=tk.NORMAL)
			self.varUnsaved = True

	# Code to save a level to the appropriate sublist in self.varRoomList.
	def savelevel (self):
		#print (self.varRoomTagmap)  # This is a debug line.
		TempDict = {}
		for index, room in enumerate (self.varRoomTagmap):
			TempDict [room] = self.varRoomTagmap [room] ["pos"]
		SortedDict = dict (sorted (TempDict.items(), key = lambda kv: kv [1], reverse = True))
		self.varRoomList [self.varNumLevel.get()] = []
		for index, room in enumerate (SortedDict):
			if (room == self.IDEnd):
				self.varRoomList [self.varNumLevel.get()].append ("0")
				self.varRoomList [self.varNumLevel.get()].append ("80")
			elif (room != self.InsertionPoint):
				self.varRoomList [self.varNumLevel.get()].append (self.varRoomTagmap [room] ["room"])
				params = int (self.varRoomTagmap [room] ["ptrduration"].get()) + self.varRoomTagmap [room] ["ptrflags"].get()
				self.varRoomList [self.varNumLevel.get()].append (hex (params) [2:])

	# Code to load the level data from the .cfg file
	def loadcfg (self):
		self.varNumRooms.set (0)
		self.varRoomList = [[], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], []]
		filecfg = open (config.outputpath + config.imagename + ".cfg", "r")  # Read-only, default method
		cfgline = " "  # Avoid an empty string because of the line below.
		while (cfgline [2:3] != 'f') and (cfgline != ""):  # Skip ahead to the line with a poke at 0xf000.  Catch EOF condition.
			cfgline = filecfg.readline()
		self.varRoomList [0].append (cfgline [7:-1])  # Strip out the EOL character at the end.
		oddaddress = True  # Used to increment self.varNumRooms every other line read
		for numlevel in range (32):
			while (cfgline != ""):  # Catch EOF condition again.
				cfgline = filecfg.readline()
				if (oddaddress):
					self.varNumRooms.set (self.varNumRooms.get() + 1)
				oddaddress = not oddaddress
				if (cfgline [7:-1] == "80"):
					break
				self.varRoomList [numlevel].append (cfgline [7:-1])
			filecfg.readline()  # Junk to us - it's the first updated poke to the level pointer table.
			filecfg.readline()  # Ditto - it's the second updated poke.
			self.varRoomList [numlevel].append (cfgline [7:-1])  # Should be "80"
		filecfg.close()

	# Code to write the level data to the .cfg file
	def savecfg (self):
		filecfg = open (config.outputpath + config.imagename + ".cfg", "w")
		filecfg.write ("[mapping]\n$0000 - $1FFF = $5000\n$2000 - $2FFF = $D000\n$3000 - $3FFF = $F000\n[macro]\n")
		addressF = 61440  # In hexadecimal, this is 0xf000, the starting address for the level data.
		if (self.varConsolidate.get() == ";\n"):  # Is the "consolidate" box unchecked?"
			addressD = 56802  # In hexadecimal, this is 0xdde2, the starting address to the beginning-of-level pointer table.
		else:
			addressD = 28472  # In hexadecimal, this is 0x6f38, the updated starting address to the level pointer table (so technically it should be renamed "address6" at some point).
			filetemp = open (config.workingpath + "consolidate.txt", "r")
			templine = filetemp.readline()
			while (templine != ""):
				filecfg.write (templine)
				templine = filetemp.readline()
			filetemp.close()
		if (self.varMusic.get() != ";\n"):
			filetemp = open (config.workingpath + "musicfix.txt", "r")
			templine = filetemp.readline()
			while (templine != ""):
				filecfg.write (templine)
				templine = filetemp.readline()
			filetemp.close()
		for numlevel in range (32):
			filecfg.write ("p " + hex (addressD) [2:] + " " + hex (addressF) [4:] + "\n")  # Low byte of Fxxx address
			addressD = addressD + 1
			filecfg.write ("p " + hex (addressD) [2:] + " " + hex (addressF) [2:4] + "\n")  # High byte of Fxxx address
			addressD = addressD + 1
			for pokevalue in self.varRoomList [numlevel]:
				filecfg.write ("p " + hex (addressF) [2:] + " " + pokevalue + "\n")
				addressF = addressF + 1
		filecfg.close()
		self.btnSave.config (state = tk.DISABLED)
		self.varUnsaved = False

	# This code makes the popup window appear.
	def dopopup (self, event):
		#print ("\nevent.widget:", event.widget)
		#print ("event.x_root, event.y_root:", event.x_root, event.y_root)
		#print ("event.x, event.y:", event.x, event.y)
		#print ("self.canvasLevel.canvasx (event.x), self.canvasLevel.canvasy (event.y):", self.canvasLevel.canvasx (event.x, gridspacing = 1), self.canvasLevel.canvasy (event.y, gridspacing = 1))
		#print ("self.canvasLevel.find_closest (self.canvasLevel.canvasx (event.x), self.canvasLevel.canvasy(event.y)) [0]:", self.canvasLevel.find_closest (self.canvasLevel.canvasx (event.x), self.canvasLevel.canvasy (event.y)) [0])
		if (event.widget == self.canvasLevel):
			self.varThisRoom.set (self.canvasLevel.find_closest (self.canvasLevel.canvasx (event.x), self.canvasLevel.canvasy (event.y)) [0])  # Which room are we dealing with here?  Save it to a variable.
		else:
			for index, roomid in enumerate (self.varRoomTagmap):
				if (roomid != self.IDEnd) and (roomid != self.InsertionPoint):
					if (self.varRoomTagmap [roomid] ["ptrroom"] == event.widget):
						self.varThisRoom.set (roomid)
		try:
			self.menuPopup.tk_popup (event.x_root, event.y_root)
		finally:
			self.menuPopup.grab_release()

	# These are the commands bound to each button in the Rooms Canvas.
	# I modified the arguments for the "skinny rooms" to set their default Length values as defined in the cartridge ROM.
	def doaddroom00 (self):
		self.addroomtolevel ("0", "3f", True)
	def doaddroom01 (self):
		self.addroomtolevel ("1", "3", True)
	def doaddroom02 (self):
		self.addroomtolevel ("2", "0", True)
	def doaddroom03 (self):
		self.addroomtolevel ("3", "b", True)
	def doaddroom04 (self):
		self.addroomtolevel ("4", "15", True)
	def doaddroom05 (self):
		self.addroomtolevel ("5", "3", True)
	def doaddroom06 (self):
		self.addroomtolevel ("6", "0", True)
	def doaddroom07 (self):
		self.addroomtolevel ("7", "0", True)
	def doaddroom08 (self):
		self.addroomtolevel ("8", "0", True)
	def doaddroom09 (self):
		self.addroomtolevel ("9", "0", True)
	def doaddroom0a (self):
		self.addroomtolevel ("a", "0", True)
	def doaddroom0b (self):
		self.addroomtolevel ("b", "0", True)
	def doaddroom0c (self):
		self.addroomtolevel ("c", "0", True)
	def doaddroom0d (self):
		self.addroomtolevel ("d", "0", True)
	def doaddroom0e (self):
		self.addroomtolevel ("e", "0", True)
	def doaddroom0f (self):
		self.addroomtolevel ("f", "0", True)
	def doaddroom10 (self):
		self.addroomtolevel ("10", "0", True)
	def doaddroom11 (self):
		self.addroomtolevel ("11", "0", True)
	def doaddroom12 (self):
		self.addroomtolevel ("12", "0", True)
	def doaddroom13 (self):
		self.addroomtolevel ("13", "0", True)
	def doaddroom14 (self):
		self.addroomtolevel ("14", "0", True)

	# Other commands
	def doflagunsaved (self):
		self.btnSave.config (state = tk.NORMAL)
		self.varUnsaved = True
	def domoveinsertionpoint (self):
		self.moveinsertionpoint (self.varThisRoom.get())
	def dodeleteroom (self):
		self.deleteroom (self.varThisRoom.get())
	def dochangelevels (self, event):
		self.savelevel()
		self.loadlevel (event)
	def dosave (self):
		self.savelevel()
		self.savecfg()

	# This code fires when the "backup" button is clicked.
	def dobackup (self):
		if (self.varUnsaved):
			if not tk.messagebox.askyesno ("You have unsaved work!", "Your work must be saved before continuing.\n\nOkay?"):  # Pop up a warning message with yes/no buttons.
				return
			else:
				self.dosave()
		filescript.write ("copy \"" + config.outputpath + config.imagename + ".cfg\" \"" + config.outputpath + config.imagename + ".cfg~\"\n")
		filescript.write ("echo Backed up \"" + config.outputpath + config.imagename + ".cfg\" to \"" + config.outputpath + config.imagename + ".cfg~\"\n")
		filescript.write (config.workingpath + ".\\bnjedit.bat\n")
		filescript.close()
		self.quit()

	# This code fires when the "make .rom" button is clicked.
	def domake (self):
		if (self.varUnsaved):
			if not tk.messagebox.askyesno ("You have unsaved work!", "Your work must be saved before continuing.\n\nOkay?"):  # Pop up a warning message with yes/no buttons.
				return
			else:
				self.dosave()
		filescript.write (config.jzintvpath + "bin2rom \"" + config.outputpath + config.imagename + ".bin\" >nul\n")
		filescript.write (config.workingpath + ".\\bnjedit.bat\n")
		filescript.close()
		self.quit()

	# This code fires when the "play .rom" button is clicked.
	def doplay (self):
		if (self.varUnsaved):
			if not tk.messagebox.askyesno ("You have unsaved work!", "Your work must be saved before continuing.\n\nOkay?"):  # Pop up a warning message with yes/no buttons.
				return
			else:
				self.dosave()
		filescript.write (config.jzintvpath + "jzintv \"" + config.outputpath + config.imagename + ".rom\" " + config.jzintvoptions + "\n")
		#filescript.write ("pause\n")
		filescript.write (config.workingpath + ".\\bnjedit.bat\n")
		filescript.close()
		self.quit()

	def dohelp (self):
		tk.messagebox.showinfo ("BNJEdit Instructions (1 of 7)", "1. Click on one of the 21 rooms in the Rooms list to add it to the current Level, below where \"Insert below\" (the Insertion Point) is displayed.")
		tk.messagebox.showinfo ("BNJEdit Instructions (2 of 7)", "2. Each room except the End has controls to:\n\n a. Flag an upcoming water hazard or set a Restart point.\n\nFrom right to left, \"N/R/!\" means:\n    i. \"!\": Display the water hazard warning sign in this room,\n   ii. \"R\": Set a Restart point,\n  iii. \"N\": Neither of the above.\n\n b. Set the Length (1-63) or Repeat count (0-63), depending on the room.")
		tk.messagebox.showinfo ("BNJEdit Instructions (3 of 7)", "3. If you click on a room within the current Level, a pop-up window will appear with options to:\n\n a. Move the Insertion Point to below that room, or\n\n b. Delete the room.\n\nYou cannot delete the End Room.  You can only move the Insertion Point back to below that room.")
		tk.messagebox.showinfo ("BNJEdit Instructions (4 of 7)", "4. The two checkboxes above the Level Selector in the bottom right corner allow you to add the following options to your ROM hack:\n\n a. Tweak the music slightly to be more consistent with the arcade version,\n\n b. Move the Room and Level pointer tables in the ROM to where there is blank space, and accordingly shift the room data in the ROM.\n\nThe latter option is to consolidate ROM space to allow for additional rooms to be created for a future release.")
		tk.messagebox.showinfo ("BNJEdit Instructions (5 of 7)", "5. Use the Save button to write to " + config.imagename + ".cfg, where your work is saved.")
		tk.messagebox.showinfo ("BNJEdit Instructions (6 of 7)", "6. Use the Backup button to make a backup copy of " + config.imagename + ".cfg to \"" + config.imagename + ".cfg~\"")
		tk.messagebox.showinfo ("BNJEdit Instructions (7 of 7)", "7. If you put your own copy of " + config.imagename + ".bin in " + config.outputpath + ":\n\n a. You can use the \"Make .rom\" button to convert the hacked ROM image to .rom format,\n\n b. You can use the \"Play .rom\" button to immediately test your levels in jzIntv.")

	def doabout (self):
		tk.messagebox.showinfo ("About BNJEdit", "BNJEdit - Bump & Jump Level Editor\n\nWritten by Michael Hayes\n\nDate Of Last Modification: October 3, 2023")

	def doquit (self):
		if (self.varUnsaved):
			if not tk.messagebox.askyesno ("You have unsaved work!", "Do you want to quit without saving your work?"):  # Pop up a warning message with yes/no buttons.
				return
		filescript.close()  # The temp script file needs to be closed so its contents will be written.
		self.quit()

app = Application()
app.master.title ("Bump & Jump Level Editor")
app.master.iconphoto (True, tk.PhotoImage (file = config.workingpath + "icon.png"))
app.master.geometry ("+0+0")
app.configure (bg = config.windowbgcolor)
filescript = open (config.workingpath + "temp.bat", "w")
filescript.write (config.hashbang + "\n")
app.mainloop()
