# -------------------------------------------------------------------------------------------------
# chip8ToEcs.py - Converts a CHIP-8 binary file to an Intellivision ECS tape suitable for
# use in Chippi

import argparse


# -------------------------------------------------------------------------------------------------
# Main entry point

def main():

    # Validate those pesky arguments
    parser = argparse.ArgumentParser(description='''CHIP-8 binary file to an Intellivision ECS 
        tape suitable for use in Chippi''')
    parser.add_argument('-c', '--color', type=int, help='Set default colour')
    parser.add_argument('-d', '--quirkDisplay', action='store_true', 
        help='Assert display quirk mode')
    parser.add_argument('-m', '--quirkMemory', action='store_true', help='Assert memory quirk mode')
    parser.add_argument('-n', '--note', type=int, help='Set default note / chord')
    parser.add_argument('-s', '--quirkShift', action='store_true', help='Assert shift quirk mode')
    parser.add_argument('-t', '--throttle', type=int, help='Set default emulator speed / throttle')
    parser.add_argument('inpPath', help='Path to source CHIP-8 program file')
    parser.add_argument('name', help='4 character name of the program')
    args = parser.parse_args()
    
    if (len(args.name) != 4):
        parser.print_usage()
        print("chip8ToEcs.py: error: argument name: incorrect length, must be 4 characters")
        quit()
        
    if (not args.name.isalnum()):
        parser.print_usage()
        print("chip8ToEcs.py: error: argument name: must be alphanumeric only")
        quit()
        
    out = open("ecs_tape_%s.ecs" % args.name.upper(), mode='wb')
    
    # Write out tape header
    out.write(b'\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA')
    out.write(b'\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\x00')
    
    # Write quirks mode to use
    quirks = 0
    if (args.quirkMemory):
        quirks += 1
    if (args.quirkShift):
        quirks += 2
    if (args.quirkDisplay):
        quirks += 4
        
    out.write(chr(quirks).encode('ascii'))
    
    # Read source file
    inp = open(args.inpPath, mode='rb')
    data = inp.read()
    inp.close()

    # Write file size 
    out.write(bytearray([len(data) // 256, len(data) % 256]))
    
    # Write program data
    out.write(data)
    
    # Compute CRC
    crc = 0
    for c in data:
        crc += c;
    out.write(bytearray([256 - (crc & 255)]))
    
    colour = args.color if (args.color) else 0
    out.write(chr(colour).encode('ascii'))

    note = args.note if (args.note) else 0
    out.write(chr(note).encode('ascii'))

    throttle = args.throttle if (args.throttle) else 7
    out.write(chr(throttle).encode('ascii'))

    # Done
    out.close()


main()
